/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util.swing;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.JViewport;
import javax.swing.SwingUtilities;

/**
 *  PopupMenuShower looks at mouse events to decide when it's time to show
 *  a JPopupMenu, and then adjusts the position of the popup if it wasn't
 *  completely on screen.
 */
public class PopupMenuShower extends MouseAdapter
  implements ActionListener
{
    public PopupMenuShower(JPopupMenu popup)
    {
        this((JTable)null, popup);
    }

    public PopupMenuShower(JTable table, JPopupMenu popup)
    {
        this.table = table;
        this.popup = popup;

        Object comp = table.getParent();

        if(comp != null && comp instanceof JViewport)
        {
            ((JViewport)comp).addMouseListener(this);
        }
    }

    public PopupMenuShower(JTree tree, JPopupMenu popup)
    {
        this.tree  = tree;
        this.popup = popup;
    }

    protected void showMenuIfPopupTrigger(MouseEvent event)
    {
        if (popup != null && popup.isPopupTrigger(event))
        {
            // if fewer than two rows are selected in the table or tree,
            // first select the row under the mouse.

            if ((table != null) && (table.getSelectedRowCount() < 2))
            {
                int row = table.rowAtPoint(new Point(event.getX(), event.getY()));

                if (row >= 0)
                {
                    table.setRowSelectionInterval(row, row);
                }
            }
            else if ((tree != null) && (tree.getSelectionCount() < 2))
            {
                int row = tree.getRowForLocation(event.getX(), event.getY());

                if (row >= 0)
                {
                    tree.setSelectionRow(row);
                }
            }

            if(popup.getComponentCount() > 0)
            {
                popup.show(event.getComponent(), event.getX(), event.getY());
                ensurePopupFullyOnScreen(event);
            }
        }
    }

    protected void ensurePopupFullyOnScreen(MouseEvent event)
    {
        Component comp = event.getComponent();
        Dimension ssz  = comp.getToolkit().getScreenSize();
        Dimension psz  = popup.getSize();
        Point     loc  = new Point(event.getX(), event.getY());

        SwingUtilities.convertPointToScreen(loc, comp);

        boolean mustFix = false;

        if ((loc.y + psz.height) > ssz.height)
        {
            loc.y -= psz.height;
            mustFix = true;
        }
        if ((loc.x + psz.width) > ssz.width)
        {
            loc.x -= psz.width;
            mustFix = true;
        }

        if (mustFix)
        {
            SwingUtilities.convertPointFromScreen(loc, comp);
            if(popup.getComponentCount() > 0)
            {
                popup.show(event.getComponent(), loc.x, loc.y);
            }
        }
    }

    @Override
    public void mousePressed(MouseEvent event)
    {
        showMenuIfPopupTrigger(event);
    }

    @Override
    public void mouseReleased(MouseEvent event)
    {
        showMenuIfPopupTrigger(event);
    }

    @Override
    public void actionPerformed(ActionEvent event)
    {
        Component comp = (Component)event.getSource();

        MouseEvent mouseEvent = new MouseEvent(comp, 0, 0, 0, 0, comp.getHeight() - 1, 0, true);

        if(popup.getComponentCount() > 0)
        {
            popup.show(comp, mouseEvent.getX(), mouseEvent.getY());
            ensurePopupFullyOnScreen(mouseEvent);
        }
    }

    public void setPopupMenu(JPopupMenu menu)
    {
        popup = menu;
    }

    public JPopupMenu getPopupMenu()
    {
        return(popup);
    }

    public JTree getTree()
    {
        return(tree);
    }

    public JTable getTable()
    {
        return(table);
    }

    private JTree      tree;
    private JTable     table;
    private JPopupMenu popup;
}

