/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:  
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util.swing;

import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.AbstractAction;
import javax.swing.Action;

/**
 *  RelayAction is a kind of Action that acts as a wrapper around another
 *  target Action that may be changed dynamically.  The RelayAction watches
 *  for when the target Action is enabled or disabled, and relays its own
 *  ActionEvents to the targetAction.
 */
public class RelayAction extends AbstractAction
  implements PropertyChangeListener
{
  /**
   *  Construct a RelayAction with no target Action.
   */
  public RelayAction()
  {
    this(null);
  }

  /**
   *  Construct a RelayAction with the specified target Action.
   */
  public RelayAction(Action targetAction)
  {
    allowEnable = true;
    setTargetAction(targetAction);
  }

  /**
   *  Override to get the value for the specified key from the target Action
   *  if there is no value assigned to the RelayAction itself.
   */
  @Override
public Object getValue(String key)
  {
    Object result = super.getValue(key);
    if ((result == null) && (targetAction != null))
    {
        result = targetAction.getValue(key);
    }
    return(result);
  }

  /**
   *  Assign this RelayAction's target Action.
   */
  public Action getTargetAction()
  {
    return(targetAction);
  }

  /**
   *  Assign this RelayAction's target Action.
   */
  public final void setTargetAction(Action newTargetAction)
  {
    if (targetAction != null)
    {
        targetAction.removePropertyChangeListener(this);
    }

    targetAction = newTargetAction;

    if (targetAction != null)
    {
        targetAction.addPropertyChangeListener(this);
    }

    setEnabledInternal();
  }

  /**
   *  Respond to changes in whether the target Action is enabled or not.
   */
  @Override
public void propertyChange(PropertyChangeEvent event)
  {
    if ((targetAction != null) && (event.getSource() == targetAction) &&
        "enabled".equals(event.getPropertyName())) {
      setEnabledInternal();
    }
  }

  /**
   *  Relay ActionEvents to the target Action.
   */
  @Override
public void actionPerformed(ActionEvent event)
  {
    if ((targetAction != null) && targetAction.isEnabled())
    {
        targetAction.actionPerformed(event);
    }
  }

  /**
   *  Return whether or not this RelayAction is enabled.
   */
  @Override
public boolean isEnabled()
  {
    return(super.isEnabled() && allowEnable && targetAction.isEnabled());
  }

  /**
   *  Override in order to control RelayAction directly.  A RelayAction is
   *  only actually enabled if its target Action is also enabled.  Enabling
   *  a RelayAction with this method has no effect on the target Action, but
   *  will enable or disable the RelayAction if the target Action is enabled.
   */
  @Override
public void setEnabled(boolean allowEnable)
  {
    this.allowEnable = allowEnable;
    setEnabledInternal();
  }

  private void setEnabledInternal()
  {
    super.setEnabled
      (allowEnable && (targetAction != null) && targetAction.isEnabled());
  }

  private boolean allowEnable;
  private Action  targetAction;
}
