/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util.swing;

import java.util.Comparator;
import java.util.Objects;

import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import modelobjects.util.DefaultComparator;
import modelobjects.util.ReverseComparator;

/**
 *  TableColumnAdapter is used to define the behavior of a table column in
 *  terms of some aspect or property of a row model class.  It is an abstract
 *  class that must be extended by subclasses to implement how to get and
 *  set the property values represented by the column in a table.
 *  <p>
 *  TableColumnAdapters can optionally specify their own TableCellRenderer
 *  and/or TableCellEditor.  Otherwise, the default classes are used.
 *  <p>
 *  TableColumnAdapters can specify their own Comparators for sorting in the
 *  table.
 */
public abstract class TableColumnAdapter
{
  protected TableColumnAdapter(String headerLabel,
                               boolean editable,
                               Class columnClass,
                               int minimumWidth,
                               int preferredWidth,
                               int maximumWidth)
  {
    this(headerLabel, editable, columnClass,
         minimumWidth, preferredWidth, maximumWidth,
         null, null);
  }

  protected TableColumnAdapter(String headerLabel,
                               boolean editable,
                               Class columnClass,
                               int minimumWidth,
                               int preferredWidth,
                               int maximumWidth,
                               TableCellRenderer cellRenderer,
                               TableCellEditor cellEditor)
  {
    this.headerLabel = headerLabel;
    this.editable    = editable;
    this.columnClass = columnClass;
    this.minWidth    = minimumWidth;
    this.prefWidth   = preferredWidth;
    this.maxWidth    = maximumWidth;
    this.cellEditor  = cellEditor;
    this.cellRenderer = cellRenderer;

    this.columnIsSortable = true;
  }

  public abstract Object getColumnValue(Object rowModel, int rowNumber);

  public abstract void   setColumnValue(Object rowModel, Object columnValue);


  public String getHeaderLabel()
  {
    return(headerLabel);
  }

  public void setHeaderLabel(String headerLabel)
  {
    this.headerLabel = headerLabel;
  }

  public boolean isColumnEditable(Object rowModel)
  {
    return(editable);
  }

  public void setColumnEditable(boolean editable)
  {
    this.editable = editable;
  }

  public Class getColumnClass()
  {
    return(columnClass);
  }

  public void setColumnClass(Class columnClass)
  {
    this.columnClass = columnClass;
  }

  public int getMinimumWidth()
  {
    return(minWidth);
  }

  public void setMinimumWidth(int w)
  {
    minWidth = w;
  }

  public int getMaximumWidth()
  {
    return(maxWidth);
  }

  public void setMaximumWidth(int w)
  {
    maxWidth = w;
  }

  public int getPreferredWidth()
  {
    return(prefWidth);
  }

  public void setPreferredWidth(int w)
  {
    prefWidth = w;
  }

  public final TableCellRenderer getCellRenderer()
  {
    return(cellRenderer);
  }

  public final void setCellRenderer(TableCellRenderer cellRenderer)
  {
    this.cellRenderer = cellRenderer;
  }

  public TableCellEditor getCellEditor()
  {
    return(cellEditor);
  }

  public void setCellEditor(TableCellEditor cellEditor)
  {
    this.cellEditor = cellEditor;
  }

  public ModelPropertyTableColumn makeModelPropertyTableColumn(int modelIndex)
  {
    return(new ModelPropertyTableColumn(modelIndex, this));
  }

  public Comparator getColumnComparator()
  {
    if (columnComparator == null)
    {
        columnComparator = DefaultComparator.getInstance();
    }
    return(columnComparator);
  }

  public void setColumnComparator(Comparator columnComparator)
  {
    this.columnComparator = columnComparator;
    this.rowComparator = new DefaultRowComparator(this);
    this.reverseRowComparator =
      ReverseComparator.makeReverseComparator(rowComparator);
  }

  public Comparator getRowComparator()
  {
    if (rowComparator == null) {
      rowComparator = new DefaultRowComparator(this);
      reverseRowComparator =
        ReverseComparator.makeReverseComparator(rowComparator);
    }
    return(rowComparator);
  }

  public void setRowComparator(Comparator rowComparator)
  {
    this.rowComparator = rowComparator;
    this.reverseRowComparator =
      ReverseComparator.makeReverseComparator(rowComparator);
  }

  public Comparator getReverseRowComparator()
  {
    return(reverseRowComparator);
  }

  public boolean isColumnSortable()
  {
    return(columnIsSortable);
  }

  public void setColumnSortable(boolean sortable)
  {
    columnIsSortable = sortable;
  }

  @Override
public String toString()
  {
    return(headerLabel);
  }

  private String                headerLabel;
  private boolean               editable;
  private Class                 columnClass;
  private int                   minWidth;
  private int                   maxWidth;
  private int                   prefWidth;

  private Comparator            columnComparator;
  private Comparator            rowComparator;
  private Comparator            reverseRowComparator;
  private boolean               columnIsSortable;

  private TableCellRenderer     cellRenderer;
  private TableCellEditor       cellEditor;


  public static class DefaultRowComparator implements Comparator
  {
    public DefaultRowComparator(TableColumnAdapter columnAdapter)
    {
      this.columnAdapter    = columnAdapter;
      this.columnComparator = columnAdapter.getColumnComparator();
    }

    @Override
    public int compare(Object row1, Object row2)
    {
      // row numbers really shouldn't matter here!
      Object cell1 = columnAdapter.getColumnValue(row1, 0);
      Object cell2 = columnAdapter.getColumnValue(row2, 0);

      return(columnComparator.compare(cell1, cell2));
    }

    @Override
    public boolean equals(Object obj)
    {
      if (obj != null && this.getClass() == obj.getClass()) {
        DefaultRowComparator that = (DefaultRowComparator)obj;
        return (this.columnAdapter == that.columnAdapter) &&
               (this.columnComparator.equals(that.columnComparator));
      }
      else {
        return false;
      }
    }
    
    @Override
    public int hashCode() {
        return Objects.hash(columnAdapter, columnComparator);
    }

    private TableColumnAdapter columnAdapter;
    private Comparator         columnComparator;
  }
}
