/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */

package modelobjects.util.swing;

import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JTable;
import javax.swing.table.TableColumnModel;

/**
 *  TableSorter is intended to be used as a MouseListener on a JTableHeader
 *  in order to interpret mouse-clicks as table sorting requests.  It can
 *  only be used with IModelTableModels, and cannot be used with JTables
 *  that use any other kind of TableModel.
 */
public class TableSorter extends MouseAdapter
{
  public TableSorter(JTable table)
  {
    this.table = table;
  }

  @Override
public void mousePressed(MouseEvent event)
  {
    popupTrigger = event.isPopupTrigger();
  }

  @Override
public void mouseReleased(MouseEvent event)
  {
    popupTrigger = event.isPopupTrigger();
  }

  @Override
public void mouseClicked(MouseEvent event)
  {
    // Catch if the popupTrigger is set on a mousePressed and mouseReleased
    // event so that we can use it here...because we don't want to perform
    // sorting on a popupTrigger event...that is exclusively for the popup
    // menu.
    //
    if (popupTrigger || (event.getClickCount() != 1))
    {
        return;
    }

    IModelTableModel dm = (IModelTableModel)table.getModel();
    TableColumnModel cm = table.getColumnModel();

    int viewColumn  = cm.getColumnIndexAtX(event.getX());
    int modelColumn = table.convertColumnIndexToModel(viewColumn);

    if (modelColumn != -1)
    {
      Object[] selectedObjects = getSelectedObjects(dm);

      try
      {
        table.getSelectionModel().setValueIsAdjusting(true);
        table.clearSelection();
        dm.sortByColumn(modelColumn);
        restoreObjectSelection(selectedObjects, dm);
      }
      finally
      {
        table.getSelectionModel().setValueIsAdjusting(false);
      }

      table.getTableHeader().repaint();
    }
  }

  protected Object[] getSelectedObjects(IModelTableModel tableModel)
  {
    int[] selectedRows = table.getSelectedRows();
    int num = selectedRows.length;
    Object[] selectedObjects = new Object[num];
    for (int i = 0; i < num; i++) {
      selectedObjects[i] = tableModel.getRowModel(selectedRows[i]);
    }
    return(selectedObjects);
  }

  protected void restoreObjectSelection(Object[]         selectedObjects,
                                        IModelTableModel tableModel)
  {
    // table selection is expected to be cleared at this point
    int num = selectedObjects.length;
    for (int i = 0; i < num; i++) {
      Object obj = selectedObjects[i];
      int row = ((obj == null) ? -1 : tableModel.getIndexOf(obj));
      if (row >= 0) {
        table.getSelectionModel().setValueIsAdjusting(true);
        table.addRowSelectionInterval(row, row);
      }
    }
  }

  private JTable  table;
  private boolean popupTrigger;
}
