/* ========================================================================
 *
 * The ModelObjects Group Software License, Version 1.0
 *
 *
 * Copyright (c) 2000-2001 ModelObjects Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        ModelObjects Group (http://www.modelobjects.com)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The name "ModelObjects" must not be used to endorse or promote
 *    products derived from this software without prior written permission.
 *    For written permission, please contact djacobs@modelobjects.com.
 *
 * 5. Products derived from this software may not be called "ModelObjects",
 *    nor may "ModelObjects" appear in their name, without prior written
 *    permission of the ModelObjects Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE MODEL OBJECTS GROUP OR ITS
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ========================================================================
 */
package modelobjects.util.swing;

import java.awt.Color;
import java.awt.Font;
import java.text.DecimalFormat;
import java.util.StringTokenizer;

import javax.swing.DefaultCellEditor;
import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

/**
 *  TimeCodeEditor is used to support hand-editing of time-codes.
 */
public class TimeCodeCellEditor extends DefaultCellEditor
{
  /**
   *  Create a TimeCodeCellEditor with the default JTextField font,
   *  that allows only values in whole seconds between the inclusive bounds.
   */
  public static TimeCodeCellEditor makeInstance(int minSeconds, int maxSeconds)
  {
    return(makeInstance(UIManager.getFont("TextField.font"),
                        true, (double)minSeconds, (double)maxSeconds));
  }

  /**
   *  Create a TimeCodeCellEditor with the specified font that allows
   *  fractions of seconds values from 0 seconds to 24 hours.
   */
  public static TimeCodeCellEditor makeInstance(Font font)
  {
    return(makeInstance(font, false, 0.0, 24*60*60.0));
  }

  /**
   *  Create a TimeCodeCellEditor with the specified font and that specifies
   *  whether to allow only whole second values, between the specified bounds.
   */
  public static TimeCodeCellEditor makeInstance(Font font,
                                                boolean wholeSeconds,
                                                double minSeconds,
                                                double maxSeconds)
  {
    JTextField textField = new JTextField(10);
    textField.setFont(font);
    textField.setHorizontalAlignment(JTextField.RIGHT);

    FieldValidator validator =
      new FieldValidator(textField, wholeSeconds, minSeconds, maxSeconds);
    textField.getDocument().addDocumentListener(validator);

    return(new TimeCodeCellEditor(textField, validator));
  }

  private TimeCodeCellEditor(JTextField textField, FieldValidator validator)
  {
    super(textField);
    this.textField = textField;
    this.validator = validator;
    setProperties();
  }

  private void setProperties() {
      setClickCountToStart(1);
      setTimeInSeconds(0.0);      // initialize to a reasonable default value
  }
  
  /**
   *  Set the time in seconds as a double.
   */
  public void setTimeInSeconds(double timeInSeconds)
  {
    if (timeInSeconds < 0)
    {
        throw(new IllegalArgumentException("value must be non-negative"));
    }
    textField.setText(getTimeString(timeInSeconds,
                                    validator.getWholeSeconds()));
  }

  /**
   *  Get the time in seconds as a double.
   */
  public double getTimeInSeconds()
  {
    String text = textField.getText();
    if (!validator.checkText(text))
    {
        throw(new NumberFormatException(text + "is not a valid number"));
    }

    if (!validator.checkRange(text))
    {
        throw(new NumberFormatException("'" + text + "' is outside the valid range of " +
                                          (long)validator.getMinSeconds() +
                                          " to " +
                                          (long)validator.getMaxSeconds()));
    }
    else
    {
        return(parseTimeIndex(text));
    }
  }

  /**
   *  Translate a more loosely formatted time-string to a double value.
   */
  public static double parseTimeIndex(String timeString)
  {
    if ((timeString == null) || (timeString.length() == 0))
    {
        return(-1);
    }

    StringTokenizer toks = new StringTokenizer(timeString, ":.", true);
    double time = 0.0;
    char lastType = '\0';

    while (toks.hasMoreTokens()) {
      String tok = toks.nextToken();
      char c1 = tok.charAt(0);

      if ((c1 == ':') || (c1 == '.')) {
        lastType = c1;
      }
      else {
        if (lastType == '.')
        {
            time = time + Double.parseDouble("0." + tok);
        }
        else
        {
            time = time * 60.0 + Double.parseDouble(tok);
        }
        lastType = '1';
      }
    }
    return(time);
  }

  /**
   *  Return a String rendering a time value in seconds.
   *  If wholeSeconds is true, returns the string formatted as hh:mm:ss.###,
   *  or as hh:mm:ss.### if whole seconds is false (in other words, with millisecond precision).
   *  Negative values for the time value are treated as zero.
   */
  public static String getTimeString(double timeInSeconds, boolean wholeSeconds)
  {
    if (wholeSeconds)
    {
        return(formatTime(timeInSeconds, format1, format2, format2));
    }
    else
    {
        return(formatTime(timeInSeconds, format1, format2, format3));
    }
  }

  public static String formatTime(double timeInSeconds,
                                  DecimalFormat hoursFormat,
                                  DecimalFormat minutesFormat,
                                  DecimalFormat secondsFormat)
  {
    if (timeInSeconds < 0.0)
    {
        return(Double.toString(timeInSeconds));
    }

    long wholeSecs  = (long)timeInSeconds;
    long hours      = (wholeSecs / 3600);
    long minutes    = ((wholeSecs / 60) % 60);
    long justSecs   = (wholeSecs % 60);
    double seconds  = (timeInSeconds - wholeSecs + justSecs);

    return(hoursFormat.format(hours) + ":" +
           minutesFormat.format(minutes) + ":" +
           secondsFormat.format(seconds));
  }

  private               JTextField      textField;
  private               FieldValidator  validator;

  private static        Color           GOOD    = Color.white;
  private static        Color           BAD     = new Color(255, 160, 160);

  private static        DecimalFormat   format1 = new DecimalFormat("#0");
  private static        DecimalFormat   format2 = new DecimalFormat("00");
  private static        DecimalFormat   format3 = new DecimalFormat("00.000");

  /////////////////////////////////////////////////////////////////////////////
  ///
  ///  inner classes
  ///
  /////////////////////////////////////////////////////////////////////////////

  static class FieldValidator implements DocumentListener
  {
    FieldValidator(JTextField textField, boolean wholeSeconds,
                   double minSeconds, double maxSeconds)
    {
      this.textField    = textField;
      this.wholeSeconds = wholeSeconds;
      this.minSeconds   = minSeconds;
      this.maxSeconds   = maxSeconds;
    }

    void checkDoc() {
      Color background = textField.getBackground();
      if (checkText(textField.getText()) &&
          checkRange(textField.getText())) {
        if (background != GOOD)
        {
            textField.setBackground(GOOD);
        }
      }
      else {
        if (background != BAD)
        {
            textField.setBackground(BAD);
        }
      }
    }

    boolean checkText(String text)
    {
      // first make sure all the characters are ok
      for (int i = 0, n = text.length(); i < n; i++) {
        char c = text.charAt(i);
        if ((c != ':') && (c != '.') && ((c < '0') || (c > '9')))
        {
            return(false);
        }
      }

      StringTokenizer toks = new StringTokenizer(text, ":.", true);
      int numCount   = 0;
      int colonCount = 0;
      int dotCount   = 0;
      char lastType  = '\0';

      while (toks.hasMoreTokens()) {
        String tok = toks.nextToken();
        char c1 = tok.charAt(0);

        if (c1 == ':') {
          if (lastType != '1')
        {
            return(false);
        }
          if ((dotCount > 0) || (colonCount > 1))
        {
            return(false);
        }
          colonCount++;
          lastType = ':';
        }
        else if (c1 == '.') {
          if (lastType != '1')
        {
            return(false);
        }
          if (dotCount > 0)
        {
            return(false);
        }
          if (wholeSeconds)
        {
            return(false);
        }
          dotCount++;
          lastType = '.';
        }
        else {
          if (lastType == '1')
        {
            return(false);
        }
          if (numCount > 4)
        {
            return(false);
        }
          if ((colonCount > 0) && (dotCount == 0)) {
            if (Integer.parseInt(tok) > 59)
            {
                return(false);
            }
          }
          numCount++;
          lastType = '1';
        }
      }

      return true;
    }


    boolean checkRange(String text)
    {
      double val = parseTimeIndex(text);
      return((val >= minSeconds) && (val <= maxSeconds));
    }

    @Override
    public void changedUpdate(DocumentEvent event) {
    }
    @Override
    public void insertUpdate(DocumentEvent event) {
      checkDoc();
    }
    @Override
    public void removeUpdate(DocumentEvent event) {
      checkDoc();
    }

    public boolean getWholeSeconds()
    {
      return wholeSeconds;
    }

    public double getMinSeconds()
    {
      return minSeconds;
    }

    public double getMaxSeconds()
    {
      return maxSeconds;
    }

    private JTextField textField;
    private boolean    wholeSeconds;
    private double     minSeconds;
    private double     maxSeconds;
  }
}
