// =====================================================================================================================
// Copyright (c) 2016. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.pojo.processor;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.TypeElement;
import javax.tools.Diagnostic.Kind;
import com.aurea.sonic.esb.annotation.util.ProcessorContext;
import com.aurea.sonic.esb.pojo.annotation.InitParameter;
import com.aurea.sonic.esb.pojo.annotation.Operation;
import com.aurea.sonic.esb.pojo.annotation.Parameter;
import com.aurea.sonic.esb.pojo.annotation.SonicPOJO;
import com.aurea.sonic.esb.pojo.processor.model.AttributeModel;
import com.aurea.sonic.esb.pojo.processor.model.OperationModel;
import com.aurea.sonic.esb.pojo.processor.model.ServiceModel;

/**
 * <!-- ========================================================================================================== -->
 * Annotation Processor
 *
 * <p>
 * Mainly, it forms the ESB Service from {@literal @SonicPOJO} annotated class and
 * write ESB Service artifacts to defined locations.
 *
 * @see InitParameter
 * @see Operation
 * @see Parameter
 * @see ESBServiceValidator
 * @see ESBServiceGenerator
 *
 * @since 10.0.7
 * <!-- --------------------------------------------------------------------------------------------------------- -->
 */
public class AnnotationProcessor extends AbstractProcessor {

    private ProcessorContext context;
    private ESBServiceGenerator generator;
    private ESBServiceValidator validator;
    private Map<Element, ServiceModel> services;
    private boolean disabled;

    /**
     * <!-- ================================================================================================== -->
     * Initializes the processing environment and constructs {@linkplain ProcessorContext}, {@linkplain ESBServiceGenerator},
     * {@linkplain ESBServiceValidator} objects from given {@code processingEnv}
     *
     * @param processingEnv processing environment
     *
     * @see AbstractProcessor#init(ProcessingEnvironment)
     * <!-- ------------------------------------------------------------------------------------------------- -->
     */
    @Override
    public synchronized void init(final ProcessingEnvironment processingEnv) {
        super.init(processingEnv);
        context = new ProcessorContext(processingEnv);
        generator = new ESBServiceGenerator(context);
        validator = new ESBServiceValidator(context);
        services = new LinkedHashMap<>();
        disabled = processingEnv.getOptions().containsKey("disabled");
    }

    @Override
    public Set<String> getSupportedOptions() {
        return Collections.singleton("disabled");
    }

    /**
     * <!-- ================================================================================================== -->
     * Returns the supported annotation types of annotation processor
     *
     * <p>
     * Supported Annotation Types:
     *
     * <ol>
     * <li>{@linkplain SonicPOJO}</li>
     * <li>{@linkplain Operation}</li>
     * <li>{@linkplain InitParameter}</li>
     * </ol>
     *
     * @return set of supported annotation types
     * <!-- ------------------------------------------------------------------------------------------------- -->
     */
    @Override
    public Set<String> getSupportedAnnotationTypes() {
        final Set<String> annotations = new LinkedHashSet<String>();
        annotations.add(SonicPOJO.class.getCanonicalName());
        annotations.add(Operation.class.getCanonicalName());
        annotations.add(InitParameter.class.getCanonicalName());
        return annotations;
    }

    /**
     * <!-- ================================================================================================== -->
     * {@inheritDoc}
     * <!-- ------------------------------------------------------------------------------------------------- -->
     */
    @Override
    public SourceVersion getSupportedSourceVersion() {
        return SourceVersion.latestSupported();
    }

    /**
     * <!-- ================================================================================================== -->
     * This method runs the following actions in order.
     *
     * <ol>
     * <li>Process the {@linkplain SonicPOJO}, {@linkplain Operation}, {@linkplain InitParameter} annotations and construct corresponding models</li>
     * <li>Validate the constructed ESB Service model via {@link ESBServiceValidator#validateService(ServiceModel)}</li>
     * <li>Generate ESB Service artifacts from ESB Service Model via {@link ESBServiceGenerator#generateService(ServiceModel)} and write generated artifacts to proper locations</li>
     * </ol>
     *
     * @param annotations supported annotations
     * @param roundEnv information about a round of annotation processing
     *
     * @return {@code true} if annotation processing is finished without any problem, otherwise {@code false}
     *
     * @see ServiceModel
     * @see OperationModel
     * @see AttributeModel
     * <!-- ------------------------------------------------------------------------------------------------- -->
     */
    @Override
    public boolean process(final Set<? extends TypeElement> annotations, final RoundEnvironment roundEnv) {
        if (disabled) {
            context.getMessager().printMessage(Kind.WARNING, "Sonic POJO annotation processor disabled");
            return true;
        }

        for (final Element annotatedElement : roundEnv.getElementsAnnotatedWith(SonicPOJO.class)) {
            final ServiceModel service = new ServiceModel(context, annotatedElement);
            services.put(service.getType(), service);
        }

        for (final Element annotatedElement : roundEnv.getElementsAnnotatedWith(Operation.class)) {
            final OperationModel operation = new OperationModel(context, annotatedElement);
            final ServiceModel serviceType = getEnclosedServiceType(annotatedElement);
            if (serviceType != null) {
                serviceType.addOperation(operation);
            }
        }

        for (final Element annotatedElement : roundEnv.getElementsAnnotatedWith(InitParameter.class)) {
            final AttributeModel attribute = new AttributeModel(context, annotatedElement);
            final ServiceModel service = getEnclosedServiceType(annotatedElement);
            if (service != null) {
                service.addAttribute(attribute);
            }
        }

        if (roundEnv.processingOver()) {
            for (final ServiceModel service : services.values()) {
                if (validator.validateService(service)) {
                    generator.generateService(service);
                }
            }
            services.clear();
        }

        return true;
    }

    private ServiceModel getEnclosedServiceType(final Element element) {
        final Element operationClass = element.getEnclosingElement();
        return services.get(operationClass);
    }

}