// =====================================================================================================================
// Copyright (c) 2016. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.pojo.processor;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Writer;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Properties;
import java.util.TreeSet;

import javax.annotation.processing.Filer;
import javax.annotation.processing.Messager;
import javax.tools.Diagnostic;
import javax.tools.FileObject;
import javax.tools.StandardLocation;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.aurea.sonic.esb.annotation.util.ProcessorContext;
import com.aurea.sonic.esb.pojo.processor.model.AttributeModel;
import com.aurea.sonic.esb.pojo.processor.model.OperationModel;
import com.aurea.sonic.esb.pojo.processor.model.ParameterModel;
import com.aurea.sonic.esb.pojo.processor.model.ServiceModel;

/**
 * <!-- ========================================================================================================== -->
 * Construct ESB Service from {@linkplain ServiceModel} and write the artifacts to specified locations.
 *
 * @since 10.0.7
 * <!-- --------------------------------------------------------------------------------------------------------- -->
 */
public class ESBServiceGenerator {

	private final Filer filer;
	private final Messager messager;
	private final String projectName;

	private static final String SONIC_PACKAGE = "sonicesb.services";
	private static final StandardLocation SONIC_LOCATION = StandardLocation.SOURCE_OUTPUT;

	/**
	 * Constructs a new object from {@code context}
	 *
	 * @param context processor context
	 */
	public ESBServiceGenerator(final ProcessorContext context) {
		this.filer = context.getFiler();
		this.messager = context.getMessager();
		this.projectName = getProjectName();
	}

	/**
	 * <!-- ================================================================================================== -->
	 * Prepares the following files from given {@code service} service model and write them to appropriate locations.
	 *
	 * <p>
	 * Files:
	 *
	 * <ol>
	 * <li>.esbstyp file</li>
	 * <li>.properties file</li>
	 * <li>service interface xml</li>
	 * </ol>
	 *
	 * @param service model of esb service
	 * <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	public void generateService(final ServiceModel service) {
		try {
			final Document esbTypeXml = createEsbtypXml(service);
			writeSonicXml(service.getName() + ".esbstyp", esbTypeXml);

			final Properties esbTypeProperties = createEsbTypeProperties(service);
			writeSonicProperties(service.getName() + ".properties", esbTypeProperties);

			final Document esbTypeInterfaceXml = createEsbtypInterfaceXml(service);
			writeSonicXml(service.getName() + "Interface.xml", esbTypeInterfaceXml);

			copyServiceIconFile(service);
        } catch (final Exception ex) {
			addError(service.getType(),
					"Error while generating " + service.getName() + " service resources: " + ex.getMessage());
		}
	}

	private void copyServiceIconFile(ServiceModel service) {
		String Icon = service.getName() + ".gif";
		try {
			final FileObject inputFile = filer.getResource(StandardLocation.SOURCE_PATH, "", Icon);
			final FileObject outputFile = filer.createResource(SONIC_LOCATION, SONIC_PACKAGE, Icon);
			InputStream inputStream = null;
			OutputStream outputStream = null;
			try {
				inputStream = inputFile.openInputStream();
				outputStream = outputFile.openOutputStream();
				byte[] buffer = new byte[1024];
				int length;
				while ((length = inputStream.read(buffer)) > 0) {
					outputStream.write(buffer, 0, length);
				}
			} finally {
				if (inputStream != null) {
					try {
						inputStream.close();
					} catch (IOException e) {
						addError(service.getType(), e.getMessage());
					}
				}
				if (outputStream != null) {
					try {
						outputStream.close();
					} catch (IOException e) {
						addError(service.getType(), e.getMessage());
					}
				}
			}
		} catch (FileNotFoundException e) {
			// if there is no Icon file, just ignore
			messager.printMessage(Diagnostic.Kind.NOTE, "There is no Service Icon File");
		} catch (IOException ex) {
			addError(service.getType(), "Error while copying Service Icon "
					+ service.getName() + " service resources: " + ex.getMessage());
		}

	}

	private void addError(final javax.lang.model.element.Element element, final String message, final Object... args) {
		final String formattedMessage = String.format(message, args);
		messager.printMessage(Diagnostic.Kind.ERROR, formattedMessage, element);
	}

	private Document createEsbtypXml(final ServiceModel service) throws Exception {
		final DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		final DocumentBuilder builder = dbf.newDocumentBuilder();
		final Document doc = builder.newDocument();

		// TODO: Refactor with Velocity

		final Element serviceType = doc.createElement("serviceType");
		serviceType.setAttribute("name", service.getName());
		serviceType.setAttribute("factoryClass", "com.sonicsw.xq.service.executor.POJOExecutor");
		serviceType.setAttribute("xmlns", "http://www.sonicsw.com/sonicxq");
		serviceType.setAttribute("xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");
		serviceType.setAttribute("xsi:schemaLocation",
				"http://www.sonicsw.com/sonicxq file:///C:/Sonic/ESB2015/schema/serviceType.xsd");
		doc.appendChild(serviceType);

		final Element validParams = doc.createElement("validParams");
		serviceType.appendChild(validParams);

		final Element init = doc.createElement("init");
		validParams.appendChild(init);

		for (final AttributeModel attribute : service.getAttributes()) {
			final Element initParam = doc.createElement("stringParam");
			initParam.setAttribute("name", attribute.getName());
			initParam.setAttribute("required", Boolean.toString(attribute.isRequired()));
			init.appendChild(initParam);

			final Element defaultValue = createDefaultValueElement(doc); 
			defaultValue.setTextContent(attribute.getDefaultValue());
			initParam.appendChild(defaultValue);
		}

		final Element initParamDelegate = doc.createElement("stringParam");
		initParamDelegate.setAttribute("name", "com.sonicsw.xq.service.executor.POJOExecutor.delegate");
		initParamDelegate.setAttribute("required", "false");
		init.appendChild(initParamDelegate);

		final Element defaultValueDelegate = createDefaultValueElement(doc); 
		defaultValueDelegate.setTextContent(service.getClassName());
		initParamDelegate.appendChild(defaultValueDelegate);

		final Element runtime = doc.createElement("runtime");
		validParams.appendChild(runtime);

		final Element operationId = doc.createElement("stringParam");
		operationId.setAttribute("name", "operationId");
		runtime.appendChild(operationId);

		final Element classLoading = doc.createElement("classLoading");
		serviceType.appendChild(classLoading);

		final Element serviceTypeCL = doc.createElement("serviceType");
		classLoading.appendChild(serviceTypeCL);

		final Element clList = doc.createElement("classpath_list");
		serviceTypeCL.appendChild(clList);

		final Element clWorkspace = doc.createElement("classpath");
		clWorkspace.setTextContent("sonicfs:///workspace/" + projectName + "/lib/custom-services-classes.jar");
		clList.appendChild(clWorkspace);

		return doc;
	}
 
	private Element createDefaultValueElement(final Document doc) { 
	    final Element defaultValue = doc.createElement("default"); 
	    defaultValue.setAttribute("edittable", "true"); 
	    defaultValue.setAttribute("editAsRef", "false"); 
	    return defaultValue; 
	} 

	private final String getProjectName() {
		try {
			final FileObject resource = filer.createResource(StandardLocation.CLASS_OUTPUT, "", ".tmp");
			Path projectPath = Paths.get(resource.toUri()).getParent().getParent();
			if ("target".equals(projectPath.getFileName().toString())) {
				projectPath = projectPath.getParent();
			}
			return projectPath.getFileName().toString();
		} catch (final IOException ex) {
			return null;
		}
	}

	private Properties createEsbTypeProperties(final ServiceModel service) throws Exception {
		final Properties properties = new Properties();

		properties.setProperty("type.name", service.getName());
		properties.setProperty("type.displayName", service.getDisplayName());
		properties.setProperty("type.shortDesc", service.getDescription());
		properties.setProperty("type.defaultInstance", service.getInstanceName());
		properties.setProperty("type.isDecision", "false");
		properties.setProperty("type.mappingMode", "1");

		String initParamNames = "";
		for (final AttributeModel attribute : service.getAttributes()) {

			properties.setProperty("initParam." + attribute.getName() + ".name", attribute.getName());
			properties.setProperty("initParam." + attribute.getName() + ".displayName", attribute.getDisplayName());
			properties.setProperty("initParam." + attribute.getName() + ".shortDesc", attribute.getDescription());
			if (attribute.isPassword()) {
				properties.setProperty("initParam." + attribute.getName() + ".type", "password");
			} else {
				properties.setProperty("initParam." + attribute.getName() + ".type", "string");
			}
			properties.setProperty("initParam." + attribute.getName() + ".required",
					Boolean.toString(attribute.isRequired()));
			properties.setProperty("initParam." + attribute.getName() + ".hidden",
					Boolean.toString(attribute.isHidden()));
			properties.setProperty("initParam." + attribute.getName() + ".readOnly",
					Boolean.toString(attribute.isReadOnly()));
			properties.setProperty("initParam." + attribute.getName() + ".enableSubstitution",
					Boolean.toString(attribute.isEnableSubstitution()));
			properties.setProperty("initParam." + attribute.getName() + ".defaultValue", attribute.getDefaultValue());

			initParamNames += attribute.getName() + ",";
		}

		final String delegateInitParam = "com.sonicsw.xq.service.executor.POJOExecutor.delegate";
		properties.setProperty("initParam." + delegateInitParam + ".name", delegateInitParam);
		properties.setProperty("initParam." + delegateInitParam + ".displayName", "POJOExecutor.delegate");
		properties.setProperty("initParam." + delegateInitParam + ".shortDesc", "src/main/sonicesb/services");
		properties.setProperty("initParam." + delegateInitParam + ".type", "string");
		properties.setProperty("initParam." + delegateInitParam + ".required", "false");
		properties.setProperty("initParam." + delegateInitParam + ".hidden", "true");
		properties.setProperty("initParam." + delegateInitParam + ".readOnly", "false");
		properties.setProperty("initParam." + delegateInitParam + ".enableSubstitution", "false");
		properties.setProperty("initParam." + delegateInitParam + ".defaultValue", service.getClassName());

		initParamNames += delegateInitParam;
		properties.setProperty("type.initParamNames", initParamNames);

		final String operationRuntimeParam = "operationId";
		properties.setProperty("runtimeParam." + operationRuntimeParam + ".name", operationRuntimeParam);
		properties.setProperty("runtimeParam." + operationRuntimeParam + ".displayName", operationRuntimeParam);
		properties.setProperty("runtimeParam." + operationRuntimeParam + ".shortDesc", "");
		properties.setProperty("runtimeParam." + operationRuntimeParam + ".type", "operationIdentifier");
		properties.setProperty("runtimeParam." + operationRuntimeParam + ".required", "true");
		properties.setProperty("runtimeParam." + operationRuntimeParam + ".hidden", "false");
		properties.setProperty("runtimeParam." + operationRuntimeParam + ".readOnly", "false");
		properties.setProperty("runtimeParam." + operationRuntimeParam + ".enableSubstitution", "false");
		properties.setProperty("runtimeParam." + operationRuntimeParam + ".defaultValue", "");

		properties.setProperty("type.runtimeParamNames", operationRuntimeParam);

		return properties;
	}

	private Document createEsbtypInterfaceXml(final ServiceModel service) throws Exception {
		final DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		final DocumentBuilder builder = dbf.newDocumentBuilder();
		final Document doc = builder.newDocument();

		final Element serviceInterface = doc.createElement("interface");
		serviceInterface.setAttribute("name", service.getName());
		serviceInterface.setAttribute("xmlns", "http://www.sonicsw.com/sonicxq");
		doc.appendChild(serviceInterface);

		final Element interfaceDocumentation = doc.createElement("documentation");
		interfaceDocumentation.setTextContent(service.getDescription());
		serviceInterface.appendChild(interfaceDocumentation);

		for (final OperationModel operation : service.getOperations()) {
			final Element operationTag = doc.createElement("operation");
			operationTag.setAttribute("name", operation.getName());
			serviceInterface.appendChild(operationTag);

			final Element operationDocumentation = doc.createElement("documentation");
			operationDocumentation.setTextContent(operation.getDescription());
			operationTag.appendChild(operationDocumentation);

			final Element input = doc.createElement("input");
			input.setAttribute("name", "Input");
			operationTag.appendChild(input);

			for (final ParameterModel parameter : operation.getParameters()) {
				final Element inputParameter = doc.createElement("parameter");
				inputParameter.setAttribute("name", parameter.getName());
				inputParameter.setAttribute("type", parameter.getInputType());
				inputParameter.setAttribute("contentType", parameter.getContentType());
				inputParameter.setAttribute("collection", Boolean.toString(parameter.isCollection()));
				inputParameter.setAttribute("xmlns:xs", "http://www.w3.org/2001/XMLSchema");
				input.appendChild(inputParameter);
			}

			final Element output = doc.createElement("output");
			output.setAttribute("name", "Output");
			operationTag.appendChild(output);

			final Element outputParameter = doc.createElement("parameter");
			outputParameter.setAttribute("name", operation.getOutputName());
			outputParameter.setAttribute("type", operation.getOutputType());
			outputParameter.setAttribute("contentType", operation.getContentType());
			outputParameter.setAttribute("collection", Boolean.toString(operation.isCollection()));
			outputParameter.setAttribute("xmlns:xs", "http://www.w3.org/2001/XMLSchema");
			output.appendChild(outputParameter);
		}

		return doc;
	}

	private void writeXmlDocument(final Document xml, final Writer out) throws Exception {
		final Transformer tf = TransformerFactory.newInstance().newTransformer();
		tf.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
		tf.setOutputProperty(OutputKeys.INDENT, "yes");
		tf.transform(new DOMSource(xml), new StreamResult(out));
	}

	private void writeProperties(final Properties properties, final Writer out) throws Exception {
		@SuppressWarnings("serial")
		final Properties sortedProperties = new Properties() {
			@Override
			public synchronized Enumeration<Object> keys() {
				return Collections.enumeration(new TreeSet<Object>(super.keySet()));
			}
		};
		sortedProperties.putAll(properties);
		sortedProperties.store(out, null);
	}

	private void writeSonicXml(final String name, final Document xml) throws Exception {
		final FileObject resource = filer.createResource(SONIC_LOCATION, SONIC_PACKAGE, name);
		try (final Writer out = resource.openWriter()) {
			writeXmlDocument(xml, out);
		}
	}

	private void writeSonicProperties(final String name, final Properties properties) throws Exception {
		final FileObject resource = filer.createResource(SONIC_LOCATION, SONIC_PACKAGE, name);
		try (final Writer out = resource.openWriter()) {
			writeProperties(properties, out);
		}
	}
}
