// =====================================================================================================================
// Copyright (c) 2016. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.annotation.util;

import javax.lang.model.element.TypeElement;
import javax.lang.model.type.ArrayType;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeKind;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.Elements;
import javax.lang.model.util.Types;

public final class ElementUtil {

	private ElementUtil() {
		// utility class
	}

	public static String classToXmlType(final Class<?> clazz, final boolean isCollection) {
		final String xmlType;

		switch (clazz.getName()) {
		case "java.lang.String":
			xmlType = "xsd:string";
			break;
		case "java.lang.Long":
			xmlType = "xsd:long";
			break;
		case "java.lang.Integer":
			xmlType = "xsd:int";
			break;
		case "java.lang.Short":
			xmlType = "xsd:short";
			break;
		case "java.lang.Boolean":
			xmlType = "xsd:boolean";
			break;
		case "java.lang.Float":
			xmlType = "xsd:float";
			break;
		case "java.lang.Double":
			xmlType = "xsd:double";
			break;
		case "java.lang.Byte":
			xmlType = isCollection ? "xsd:hexBinary" : "xsd:byte";
			break;
		case "java.math.BigInteger":
			xmlType = "xsd:integer";
			break;
		case "java.math.BigDecimal":
			xmlType = "xsd:decimal";
			break;
		case "java.net.URL":
			xmlType = "xsd:string";
			break;
		case "java.util.UUID":
			xmlType = "xsd:string";
			break;
		case "java.util.Date":
			xmlType = "xsd:dateTime";
			break;
		default:
			xmlType = "xsd:any";
		}

		return xmlType;
	}

	public static Class<?> classFromType(final Types types, final TypeMirror type) {
		final Class<?> javaType;
		if (type.getKind() == TypeKind.VOID) {
			javaType = Void.class;
		} else if (type.getKind() == TypeKind.ARRAY) {
			final ArrayType arrayType = (ArrayType) type;
			final TypeElement typeElement = typeElementFromType(types, arrayType.getComponentType());
			javaType = classFromJavaType(typeElement.getQualifiedName().toString());
		} else {
			final TypeElement typeElement = typeElementFromType(types, type);
			javaType = classFromJavaType(typeElement.getQualifiedName().toString());
		}

		return javaType;
	}

	public static TypeElement typeElementFromType(final Types types, final TypeMirror type) {
		final TypeElement typeElement;

		if (type.getKind().isPrimitive()) {
			typeElement = types.boxedClass(types.getPrimitiveType(type.getKind()));
		} else if (type.getKind() == TypeKind.DECLARED) {
			final DeclaredType returnType = (DeclaredType) type;
			typeElement = (TypeElement) returnType.asElement();
		} else {
			throw new IllegalArgumentException("Invalid type kind " + type.getKind());
		}

		return typeElement;
	}

	public static Class<?> classFromJavaType(final String javaType) {
		try {
			return Class.forName(javaType);
		} catch (final ClassNotFoundException ex) {
			return null;
		}
	}

	public static boolean implementsInterface(final Types types, final Elements elements, final TypeMirror type,
			final String interfaceClass) {
		final TypeMirror typeOfE = elements.getTypeElement(interfaceClass).asType();
		final boolean isSubTypeOfE = types.isSubtype(type, typeOfE);
		return isSubTypeOfE;
	}
}
