// =====================================================================================================================
// Copyright (c) 2016. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.connect.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * The {@linkplain SonicConnect} annotation is used in conjunction with REST services and clients annotated with the JAX-RS 2.0 annotations API.
 * The  annotation indicates that a class is intended to for hosting in Sonic. The annotation interface has a number of attributes.
 * The {@code client} attribute(default {@code false}) indicates whether a REST client of server will be created.
 * <p>
 * REST clients hosted in Sonic are used to consume external REST services. An other option to JAXR-RS 2.0 annotations, is to generated REST clients
 * for Sonic hosting from WADL contracts. REST clients <b><i>only annotate an interface</i></b>, the annotation processor generates all the
 * artifacts needed for itinerary use. REST clients present a friendly interface to itinerary developers, annotated methods and parameters appear as 
 * as operations and parameters in itinerary mapping.
 * <p>
 * REST services have implementation logic of course. Hosted in Sonic have at their disposal all the standard capabilities provided by CXF(consumption from the outside via REST, data binding, 
 * 3rd jars, etc etc). In addition, REST services may be injected with Sonic service lifecycle and context access for hybrid capabilities including
 * itinerary integration and exposure etc.
 * <p>
 * Finally, REST services and client can be used together on the Sonic bus distributed across ESB containers(or, setups that map to separate JVMs,
 * separate docker containers, separate physical hosts, Sonic routing nodes etc).
 * <p>
 *
 * <h3>Service example:</h3>
 * <pre>
 * import javax.ws.rs.GET;
 * import javax.ws.rs.Path;
 * import javax.ws.rs.PathParam;
 * import javax.ws.rs.Produces;
 * import javax.ws.rs.core.MediaType;
 * import javax.ws.rs.core.Response;
 *
 * import com.aurea.sonic.esb.connect.annotation.SonicConnect;
 *
 * <strong>&#64;SonicConnect(baseUrl = "http://0.0.0.0:8080/Hello")</strong>
 * &#64;Path("hello")
 * &#64;Produces(MediaType.APPLICATION_XML)
 * public class HelloService {
 *
 * 	&#64;GET
 *	public Response sayHello() {
 *		final StringBuilder xml = new StringBuilder("&lt;response&gt;");
 *		final String hello = "&lt;message&gt;Hello everyone!&lt;/message&gt;";
 * 		xml.append(hello);
 *		xml.append("&lt;/response&gt;");
 *		return Response.ok(xml.toString()).build();
 *	}
 *
 *	&#64;GET
 *	&#64;Path("{name}")
 *	public Response saySpecialHello(&#64;PathParam("name") final String name) {
 *		final StringBuilder xml = new StringBuilder("&lt;response&gt;");
 *		final String hello = "&lt;message&gt;Special hello to " + name + "!&lt;/message&gt;";
 *		xml.append(hello);
 *		xml.append("&lt;/response&gt;");
 *		return Response.ok(xml.toString()).build();
 *	}
 *
 * }
 * </pre>
 *
 * <h2>Client Example:</h2>
 * <pre>
 * import javax.ws.rs.GET;
 * import javax.ws.rs.Path;
 * import javax.ws.rs.PathParam;
 * import javax.ws.rs.core.Response;
 *
 * import com.aurea.sonic.esb.connect.annotation.SonicConnect;
 *
 * <strong>&#64;SonicConnect(baseUrl = "http://0.0.0.0:8080/Hello", client = true)</strong>
 * &#64;Path("hello")
 * public interface HelloClient {
 *
 *	&#64;GET
 *	Response sayHello();
 *
 *	&#64;GET
 *	&#64;Path("{name}")
 *	Response saySpecialHello(&#64;PathParam("name") String name);
 *
 * }
 * </pre>
 *
 * @since 10.0.7
 */
@Documented
@Inherited
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.CLASS)
public @interface SonicConnect {

	/**
	 * display name (default: simple class name)
	 * @return name of sonic connect service
	 */
	String name() default "";

	/**
	 * comma delimited tags (e.g: REST,JAXRS). This is <i>optional</i>.
	 * @return tags of sonic connect service
	 */
	String tags() default "";

	/**
	 * baseUrl is <i>required</i><br>
	 * Should have 'http(s)' scheme, server address, port and path<br>
	 * (e.g: http://0.0.0.0:8080/rest)
	 * @return base url of sonic connect service
	 *
	 */
	String baseUrl();

	/**
	 * set to <i>true</i> for clients. Clients only provide interfaces.
	 * @return {@code true} if it is client, otherwise {@code false}
	 */
	boolean client() default false;
	
	/**
	 * Path to Spring template configuration used to generate spring.xml
	 * @return relative path to source folder to Spring template file
	 */
	String springConfigFilePath() default "";

}