// =====================================================================================================================
// Copyright (c) 2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.connect.processor;

import java.io.IOException;
import java.io.Writer;
import java.util.Date;
import java.util.List;

import javax.tools.Diagnostic;
import javax.tools.FileObject;
import javax.tools.StandardLocation;

import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import com.aurea.sonic.esb.annotation.util.ProcessorContext;
import com.aurea.sonic.esb.annotation.util.VelocityConfigurator;
import com.aurea.sonic.esb.connect.processor.model.ConnectModel;
import com.aurea.sonic.esb.connect.processor.model.ResourceModel;

/**
 * <!-- ========================================================================================================== -->
 * Construct SonicConnect Service from {@linkplain ConnectModel} and write the artifacts to specified locations.
 *
 * @since 10.0.7
 * <!-- --------------------------------------------------------------------------------------------------------- -->
 */
public class ConnectServiceGenerator {

	private final ProcessorContext context;

	private static final String ESPOSE_ESB_TEMPLATE = "templates/expose/esb.vm";
	private static final String EXPOSE_PROPERTIES_TEMPLATE = "templates/expose/properties.vm";
	private static final String INVOKE_ESB_TEMPLATE = "templates/invoke/esb.vm";
	private static final String INVOKE_PROPERTIES_TEMPLATE = "templates/invoke/properties.vm";
	private static final String CONFIG_SPRING_TEMPLATE = "templates/config/spring.vm";
	private static final String CONNECT_FOLDER = "sonicesb.connect";
	private static final StandardLocation OUTPUT_LOCATION = StandardLocation.SOURCE_OUTPUT;

	/**
	 * Sets the processor context and initializes the velocity configuration.
	 *
	 * @param context processor context
	 *
	 * @see VelocityConfigurator#init()
	 */
	public ConnectServiceGenerator(final ProcessorContext context) {
		this.context = context;
		VelocityConfigurator.init();
	}

	/**
	 * <!-- ================================================================================================== -->
	 * Create SonicConnect conventions from {@code service} parameter according to {@code service} type. <br>
	 *
	 * .esbcamel and .properties files are generated according to template decided by service type.
	 *
	 * @param service sonic connect model
	 *
	 * @return {@code true} if generating service is successful, otherwise {@code false}
	 *
	 * @see ConnectModel#isClient()
	 * <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	public boolean generateService(final ConnectModel service) {
		try {
			if (service.isClient()) {
				generateInvokeService(service);
			} else {
				generateExposeService(service);
			}
		} catch (final Exception ex) {
			addError(service.getType(), "Error while generating '" + service.getClassName()
					+ "' SonicConnect resources: " + ex.getMessage());
			return false;
		}

		return true;
	}

	/**
	 * <!-- ================================================================================================== -->
	 * Generates spring config file with new camel routes from list of connect model <br>
	 * and write the configuration file to appropriate locations.
	 *
	 * This spring config file contains Spring bean declarations, Camel context with routes, Jetty configuration.
	 *
	 * @param services list of sonic connect model
	 * <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	public void generateSpringConfig(final List<ConnectModel> services) {
	    try {
			final String configFolder = CONNECT_FOLDER + ".config";

			boolean hasServer = false;
			for (final ConnectModel service : services) {
				if (!service.isClient()) {
					hasServer = true;
					break;
				}
			}

			final VelocityContext model = retrieveModel();
			model.put("services", services);
			model.put("server", hasServer);

			final Template springTemplate = Velocity.getTemplate(CONFIG_SPRING_TEMPLATE);
			final FileObject springResource = context.getFiler().createResource(OUTPUT_LOCATION, configFolder,
					"spring.xml");
			try (final Writer out = springResource.openWriter()) {
				springTemplate.merge(model, out);
			}

		} catch (final Exception ex) {
			addError(null, "Error while generating SonicConnect spring.xml resource: " + ex.getMessage());
		}
	}

	private void generateInvokeService(final ConnectModel service) throws Exception {
		final String serviceFolder = CONNECT_FOLDER + ".services.invoke";

		final VelocityContext model = initModel(service);
		model.put("resources", service.getResources());

		final Template propertiesTemplate = Velocity.getTemplate(INVOKE_PROPERTIES_TEMPLATE);
		mergePropertiesTemplate(model, propertiesTemplate, service, serviceFolder);

		for (final ResourceModel resource : service.getResources()) {
			model.put("resource", resource);

			final Template esbTemplate = Velocity.getTemplate(INVOKE_ESB_TEMPLATE);
			final FileObject esbResource = context.getFiler().createResource(OUTPUT_LOCATION, serviceFolder,
					service.getClassName() + "." + resource.getName() + ".esbcamel");
			try (final Writer out = esbResource.openWriter()) {
				esbTemplate.merge(model, out);
			}
		}
	}

	private void generateExposeService(final ConnectModel service) throws Exception {
		final String serviceFolder = CONNECT_FOLDER + ".services.expose";

		final VelocityContext model = initModel(service);

		final Template esbTemplate = Velocity.getTemplate(ESPOSE_ESB_TEMPLATE);
		final FileObject esbResource = context.getFiler().createResource(OUTPUT_LOCATION, serviceFolder,
				service.getClassName() + ".esbcamel");
		try (final Writer out = esbResource.openWriter()) {
			esbTemplate.merge(model, out);
		}

		final Template propertiesTemplate = Velocity.getTemplate(EXPOSE_PROPERTIES_TEMPLATE);
		mergePropertiesTemplate(model, propertiesTemplate, service, serviceFolder);
	}

	private void mergePropertiesTemplate(final VelocityContext model, final Template propertiesTemplate, final ConnectModel service, final String serviceFolder) throws IOException {
		final FileObject propertiesResource = context.getFiler().createResource(OUTPUT_LOCATION, serviceFolder,
				service.getClassName() + ".properties");
		try (final Writer out = propertiesResource.openWriter()) {
			propertiesTemplate.merge(model, out);
		}
	}

	private static VelocityContext initModel(final ConnectModel service) {
		final VelocityContext model = retrieveModel();
		model.put("class", service.getClassName());
		model.put("tags", service.getTags());
		model.put("uri-alias", service.getName());
		model.put("base-url", service.getBaseUrl());
		return model;
	}

	private static VelocityContext retrieveModel() {
		final VelocityContext model = new VelocityContext();
		model.put("tool", "connect-processor");
		model.put("time-stamp", new Date().toString());
		model.put("placeholder-begins", "${");
		model.put("placeholder-ends", "}");
		return model;
	}

	private void addError(final javax.lang.model.element.Element element, final String message, final Object... args) {
		final String formattedMessage = String.format(message, args);
		context.getMessager().printMessage(Diagnostic.Kind.ERROR, formattedMessage, element);
	}

}
