// =====================================================================================================================
// Copyright (c) 2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.connect.processor;

import java.net.URI;
import java.net.URISyntaxException;

import javax.lang.model.element.Element;
import javax.tools.Diagnostic;

import com.aurea.sonic.esb.annotation.util.ProcessorContext;
import com.aurea.sonic.esb.connect.processor.model.ConnectModel;

/**
 * <!-- ========================================================================================================== -->
 *  A validator that checks the {@linkplain ConnectModel} against predefined validation rules
 *
 * @since 10.0.7
 * <!-- --------------------------------------------------------------------------------------------------------- -->
 */
public class ConnectServiceValidator {

	private final ProcessorContext context;

	public ConnectServiceValidator(final ProcessorContext context) {
		this.context = context;
	}

	/**
	 * <!-- ================================================================================================== -->
	 * Validates the {@linkplain ConnectModel#getBaseUrl()} of {@code service} with the following rules
	 *
	 * <ol>
	 * <li>Do null-check</li>
	 * <li>Checks the protocol is http</li>
	 * <li>Checks the existence of path</li>
	 * </ol>
	 *
	 * @param service sonic connect model
	 *
	 * @return {@code true}  if service is valid, otherwise {@code false}
	 * <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	public boolean validateService(final ConnectModel service) {
		if (!validateBaseUrl(service)) {
			return false;
		}

		return true;
	}

	private boolean validateBaseUrl(final ConnectModel service) {
		if (service.getBaseUrl() == null || "".equals(service.getBaseUrl())) {
			addError(service.getType(), service.getClassName() + ": baseUrl is required (e.g: http://0.0.0.0:8080/)");
			return false;
		}

		try {
			final URI uri = new URI(service.getBaseUrl());
			if (uri.getScheme() == null || !uri.getScheme().toLowerCase().startsWith("http")) {
				addError(service.getType(), service.getClassName()
						+ ": baseUrl should have 'http(s)' scheme (e.g: http://0.0.0.0:8080/)");
				return false;
			}

			if (uri.getPath() == null || uri.getPath().isEmpty()) {
				addError(service.getType(), service.getClassName()
						+ ": baseUrl should have '/' path (e.g: http://0.0.0.0:8080/)");
				return false;
			}
		} catch (final URISyntaxException ex) {
			addError(service.getType(), service.getClassName() + ": invalid baseUrl (" + ex.getMessage() + ")");
			return false;
		}

		return true;
	}

	private void addError(final Element element, final String message, final Object... args) {
		final String formattedMessage = String.format(message, args);
		context.getMessager().printMessage(Diagnostic.Kind.ERROR, formattedMessage, element);
	}

}