// =====================================================================================================================
// Copyright (c) 2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.connect.processor;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.TypeElement;

import com.aurea.sonic.esb.annotation.util.ProcessorContext;
import com.aurea.sonic.esb.connect.annotation.SonicConnect;
import com.aurea.sonic.esb.connect.processor.model.ConnectModel;
import javax.tools.Diagnostic.Kind;

/**
 * <!-- ========================================================================================================== -->
 * Annotation Processor of {@literal @}{@linkplain SonicConnect}
 *
 * <p>
 * Mainly, it forms the SonicConnect Service from {@literal @}{@linkplain SonicConnect} annotated class and
 * write related artifacts to defined locations.
 *
 * @since 10.0.7
 * <!-- --------------------------------------------------------------------------------------------------------- -->
 */
public class AnnotationProcessor extends AbstractProcessor {

	private ProcessorContext context;
	private ConnectServiceGenerator generator;
	private ConnectServiceValidator validator;
	private Map<Element, ConnectModel> services;
	private boolean disabled;

	/**
	 * <!-- ================================================================================================== -->
	 * Initializes the processing environment and constructs {@linkplain ProcessorContext}, {@linkplain ConnectServiceGenerator},
	 * {@linkplain ConnectServiceValidator} objects from given {@code processingEnv}
	 *
	 * @param processingEnv processing environment
	 *
	 * @see AbstractProcessor#init(ProcessingEnvironment)
	 * <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	@Override
	public synchronized void init(final ProcessingEnvironment processingEnv) {
		super.init(processingEnv);
		context = new ProcessorContext(processingEnv);
		generator = new ConnectServiceGenerator(context);
		validator = new ConnectServiceValidator(context);
		services = new LinkedHashMap<>();
		disabled = processingEnv.getOptions().containsKey("disabled");
	}

	@Override
	public Set<String> getSupportedOptions() {
		return Collections.singleton("disabled");
	}

	/**
	 * <!-- ================================================================================================== -->
	 * Returns the set of annotation processor's supported annotation types
	 *
	 * Supported Annotation Types:
	 *
	 * <ol>
	 * <li>{@linkplain SonicConnect}</li>
	 * </ol>
	 *
	 * @return set of supported annotation types
	 * <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	@Override
	public Set<String> getSupportedAnnotationTypes() {
		final Set<String> annotations = new LinkedHashSet<String>();
		annotations.add(SonicConnect.class.getCanonicalName());
		return annotations;
	}

	/**
	 * <!-- ================================================================================================== -->
	 * {@inheritDoc}
	 * <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	@Override
	public SourceVersion getSupportedSourceVersion() {
		return SourceVersion.latestSupported();
	}

	/**
	  * <!-- ================================================================================================== -->
	  * Process the {@linkplain SonicConnect} annotation and runs the following actions in order.
	  *
	  * <ol>
	  * <li>Construct corresponding {@linkplain ConnectModel} object</li>
	  * <li>Validate the constructed SonicConnect models via {@link ConnectServiceValidator#validateService(ConnectModel)}</li>
	  * <li>Generate Invoke Service or Expose service settings (.esbcamel and .properties files ) from Sonic Connect Model via {@link ConnectServiceGenerator#generateService(ConnectModel)}</li>
	  * <li>Generate Spring Config XML file from created artifacts via {@linkplain ConnectServiceGenerator#generateSpringConfig(List)}</li>
	  * </ol>
	  *
	  * @param annotations supported annotations
	  * @param roundEnv information about a round of annotation processing
	  *
	  * @return {@code true} if annotation processing is finished without any problem
	  * <!-- ------------------------------------------------------------------------------------------------- -->
	  */
	@Override
	public boolean process(final Set<? extends TypeElement> annotations, final RoundEnvironment roundEnv) {
		if (disabled) {
		    context.getMessager().printMessage(Kind.WARNING, "Sonic Connect annotation processor disabled");
		    return true;
		}

		for (final Element annotatedElement : roundEnv.getElementsAnnotatedWith(SonicConnect.class)) {
			final ConnectModel service = new ConnectModel(context, annotatedElement);
			services.put(service.getType(), service);
		}

		if (roundEnv.processingOver()) {
			final List<ConnectModel> generatedServices = new ArrayList<>();
			for (final ConnectModel service : services.values()) {
				if (validator.validateService(service)) {
					if (generator.generateService(service)) {
						generatedServices.add(service);
					}
				}
			}
			if (!generatedServices.isEmpty()) {
				generator.generateSpringConfig(generatedServices);
			}
			services.clear();
		}

		return true;
	}

}