// =====================================================================================================================
// Copyright (c) 2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.connect.processor.model;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeKind;
import javax.lang.model.util.ElementFilter;
import javax.tools.Diagnostic.Kind;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.MatrixParam;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;

import com.aurea.sonic.esb.annotation.util.ElementUtil;
import com.aurea.sonic.esb.annotation.util.ProcessorContext;
import com.aurea.sonic.esb.connect.annotation.SonicConnect;

/**
 * <!-- ========================================================================================================== -->
 * Resource Model of {@linkplain SonicConnect}
 *
 * <!-- --------------------------------------------------------------------------------------------------------- -->
 */
public class ResourceModel {

	public static final String ROOT = "root";

	private final Element type;

	private String name;

	private String path;

	private String qualifiedPath;

	private final Set<ParameterModel> parameters = new LinkedHashSet<>();

	public ResourceModel(final ProcessorContext context, final Element element, final String parentPath) {
		this.type = element;

		final Path pathAnnotation = element.getAnnotation(Path.class);

		if (parentPath == null) {
			this.name = ROOT;
			this.path = sanitizeAnnotationPath(pathAnnotation, null);

			registerRootParameters(context, element);
		} else if (element instanceof ExecutableElement) {
			this.name = element.getSimpleName().toString();
			this.path = sanitizeAnnotationPath(pathAnnotation, parentPath);
		}

		registerPathParameters(context);
		registerMatrixParameters(context);
		registerQueryParameters(context);

		this.qualifiedPath = createPathWithParameters(this.path);
	}

	private String createPathWithParameters(final String path) {
		final StringBuilder sb = new StringBuilder(path);

		// add matrix params to path
		String paramSeparator = ";";
		for (final ParameterModel p : parameters) {
			if (p.getKind() == ParameterKind.MATRIX) {
				sb.append(paramSeparator).append(p.getName()).append("=").append("{").append(p.getName()).append("}");
			}
		}

		// add query params to path
		paramSeparator = "?";
		for (final ParameterModel p : parameters) {
			if (p.getKind() == ParameterKind.QUERY) {
				sb.append(paramSeparator).append(p.getName()).append("=").append("{").append(p.getName()).append("}");
				paramSeparator = "&";
			}
		}

		return sb.toString();
	}

	private String sanitizeAnnotationPath(final Path path, final String parentPath) {
		final String value = path != null ? path.value() : null;
		final String result;

		if (value != null) {
			if (value.isEmpty()) {
				result = "";
			} else {
				// Remove "/" from beginning
				if (value.startsWith("/")) {
					result = value.length() == 1 ? "" : sanitizePathParam(value.substring(1));
				} else {
					result = sanitizePathParam(value);
				}
			}
		} else {
			result = "";
		}

		if (parentPath == null) {
			return "/" + result;
		} else if (!parentPath.endsWith("/") && !result.isEmpty()) {
			return parentPath + "/" + result;
		} else {
			return parentPath + result;
		}
	}

	private String sanitizePathParam(final String value) {
		// remove path param options after ":"
		final StringBuilder sb = new StringBuilder();
		final String[] paramList = value.split("/");
		for (int i = 0; i < paramList.length; i++) {
			if (paramList[i].isEmpty()) {
				continue;
			}
			if (paramList[i].startsWith("{")) {
				final String sanitizedParam;
				final int paramConfig = paramList[i].indexOf(":");
				if (paramConfig > 0) {
					sanitizedParam = paramList[i].substring(0, paramConfig).trim() + "}";
				} else {
					sanitizedParam = paramList[i];
				}
				sb.append(sanitizedParam);
			} else {
				sb.append(paramList[i]);
			}
			if (i < paramList.length - 1) {
				sb.append("/");
			}
		}
		return sb.toString();
	}

	private void addParameter(final ProcessorContext context, final String paramName, final Element paramElement,
			final ParameterKind kind) {
		final String paramType;

		if (paramElement != null) {
			final boolean isCollection = paramElement.asType().getKind() == TypeKind.ARRAY;
			final Class<?> javaType = ElementUtil.classFromType(context.getTypes(), paramElement.asType());
			paramType = ElementUtil.classToXmlType(javaType, isCollection);
		} else {
			paramType = ElementUtil.classToXmlType(Object.class, false);
		}

		final ParameterModel parameter = new ParameterModel(paramName, paramType, kind);
		parameters.add(parameter);
	}

	private <A extends Annotation> Map<String, Element> getParamMap(final Class<A> annotationClass) {
		final Map<String, Element> paramMap = new HashMap<>();
		final List<? extends Element> paramElements;
		if (ROOT.equals(name)) {
			final TypeElement classElement = (TypeElement) type;
			paramElements = ElementFilter.fieldsIn(classElement.getEnclosedElements());
		} else {
			final ExecutableElement methodElement = (ExecutableElement) type;
			paramElements = methodElement.getParameters();
		}
		for (final Element e : paramElements) {
			final Annotation annotation = e.getAnnotation(annotationClass);
			if (annotation != null) {
				final Method valueMethod;
				final String value;
				try {
					valueMethod = annotationClass.getMethod("value");
					value = (String) valueMethod.invoke(annotation);
				} catch (final NoSuchMethodException | SecurityException | IllegalAccessException
						| IllegalArgumentException | InvocationTargetException ex) {
					throw new RuntimeException("Invalid method", ex);
				}

				paramMap.put(value, e);
			}
		}
		return paramMap;
	}

	private void registerRootParameters(final ProcessorContext context, final Element element) {
		// root resource methods (GET, POST, PUT, DELETE) without @Path
		final List<ExecutableElement> methods = ElementFilter.methodsIn(element.getEnclosedElements());
		for (final ExecutableElement method : methods) {
			if (method.getAnnotation(Path.class) == null) {
				if (method.getAnnotation(GET.class) != null || method.getAnnotation(POST.class) != null
						|| method.getAnnotation(PUT.class) != null || method.getAnnotation(DELETE.class) != null) {
					final ResourceModel innerMethod = new ResourceModel(context, method, this.path);
					this.parameters.addAll(innerMethod.getParameters());
				}
			}
		}
	}

	private void registerPathParameters(final ProcessorContext context) {
		// list all path parameters on path
		final List<String> pathParameters = new ArrayList<>();
		final String[] pathList = path.split("/");
		for (final String p : pathList) {
			if (p.startsWith("{")) {
				final int paramClose = p.indexOf("}");
				if (paramClose > 0) {
					pathParameters.add(p.substring(1, paramClose).trim());
				} else {
					context.getMessager().printMessage(Kind.ERROR, "Invalid path param '" + path + "'", type);
				}
			}
		}

		// list all declared @PathParam on class/method
		final Map<String, Element> pathParamMap = getParamMap(PathParam.class);

		// create a model for each parameter with field data type
		for (final String p : pathParameters) {
			addParameter(context, p, pathParamMap.get(p), ParameterKind.TEMPLATE);
		}
	}

	private void registerMatrixParameters(final ProcessorContext context) {
		// list all declared @MatrixParam on class/method
		final Map<String, Element> pathParamMap = getParamMap(MatrixParam.class);

		for (final Entry<String, Element> p : pathParamMap.entrySet()) {
			addParameter(context, p.getKey(), p.getValue(), ParameterKind.MATRIX);
		}
	}

	private void registerQueryParameters(final ProcessorContext context) {
		// list all declared @QueryParam on class/method
		final Map<String, Element> pathParamMap = getParamMap(QueryParam.class);

		for (final Entry<String, Element> p : pathParamMap.entrySet()) {
			// create a model for each parameter with field data type
			addParameter(context, p.getKey(), p.getValue(), ParameterKind.QUERY);
		}
	}

	public Element getType() {
		return type;
	}

	public Set<ParameterModel> getParameters() {
		return Collections.unmodifiableSet(parameters);
	}

	public void setName(final String name) {
		this.name = name;
	}

	public String getName() {
		return name;
	}

	public void setPath(final String path) {
		this.path = path;
	}

	public String getPath() {
		return path;
	}

	public String getQualifiedPath() {
		return qualifiedPath;
	}

	public void setQualifiedPath(final String qualifiedPath) {
		this.qualifiedPath = qualifiedPath;
	}

	/**
	 * <!-- ================================================================================================== -->
	* {@inheritDoc}
	* <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((qualifiedPath == null) ? 0 : qualifiedPath.hashCode());
		return result;
	}

	/**
	 * <!-- ================================================================================================== -->
	* {@inheritDoc}
	* <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	@Override
	public boolean equals(final Object obj) {
		if (this == obj)
		{
			return true;
		}
		if (obj == null)
		{
			return false;
		}
		if (getClass() != obj.getClass())
		{
			return false;
		}
		final ResourceModel other = (ResourceModel) obj;
		if (qualifiedPath == null) {
			if (other.qualifiedPath != null)
			{
				return false;
			}
		} else if (!qualifiedPath.equals(other.qualifiedPath))
		{
			return false;
		}
		return true;
	}

	/**
	 * <!-- ================================================================================================== -->
	* {@inheritDoc}
	* <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	@Override
	public String toString() {
		return "ResourceModel [type=" + type + ", name=" + name + ", path=" + path + ", qualifiedPath=" + qualifiedPath
				+ ", parameters=" + parameters + "]";
	}

}
