// =====================================================================================================================
// Copyright (c) 2017. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be
// covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and
// patent-pending products
// included in the software, software updates, their related technology and services with one or
// more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent
// Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive
// listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise,
// the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the
// products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.connect.processor.model;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import javax.annotation.processing.Filer;
import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.util.ElementFilter;
import javax.tools.Diagnostic;
import javax.tools.StandardLocation;
import javax.ws.rs.Path;
import com.aurea.sonic.esb.annotation.util.ElementUtil;
import com.aurea.sonic.esb.annotation.util.ProcessorContext;
import com.aurea.sonic.esb.connect.annotation.SonicConnect;

/**
 * Model of {@linkplain SonicConnect}
 */
public class ConnectModel {

    private final Element type;

    private final String className;

    private String name;

    private String tags;

    private String baseUrl;

    private boolean client;

    private boolean xqAware;

    private String springConfig;

    private final Set<ResourceModel> resources = new LinkedHashSet<>();

    /**
     * Creates Connect Model from {@code context} and {@code element}
     *
     * @param context processor context
     * @param element element
     */
    public ConnectModel(final ProcessorContext context, final Element element) {
        this.type = element;

        final TypeElement classElement = (TypeElement) element;
        this.className = classElement.getQualifiedName().toString();

        final SonicConnect serviceAnnotation = element.getAnnotation(SonicConnect.class);
        this.name = !"".equals(serviceAnnotation.name()) ? serviceAnnotation.name()
                : classElement.getSimpleName().toString();
        this.tags = serviceAnnotation.tags();
        this.baseUrl = serviceAnnotation.baseUrl();
        this.client = serviceAnnotation.client();
        this.xqAware = ElementUtil.implementsInterface(context.getTypes(), context.getElements(),
                classElement.asType(), "com.progress.sonic.esb.camel.XQInitContextAware");
        if (!serviceAnnotation.springConfigFilePath().isEmpty()) {
            String springConfigFilePath = serviceAnnotation.springConfigFilePath();
            try {
                this.setSpringConfig(
                        loadSpringConfiguration(context.getFiler(), springConfigFilePath));
            } catch (IOException e) {
                context.getMessager().printMessage(Diagnostic.Kind.ERROR,
                        String.format("Cannot read Spring configuration from %s, error message: %s",
                                springConfigFilePath, e.getMessage()),
                        element);
            }
        }
        final ResourceModel rootResource = new ResourceModel(context, classElement, null);
        resources.add(rootResource);

        final List<ExecutableElement> methods =
                ElementFilter.methodsIn(classElement.getEnclosedElements());
        for (final ExecutableElement method : methods) {
            // contains @Path annotation (new routes)
            final Path methodPath = method.getAnnotation(Path.class);
            if (methodPath != null && !"".equals(methodPath.value())) {
                final ResourceModel resource =
                        new ResourceModel(context, method, rootResource.getPath());
                if (!resources.contains(resource)) {
                    resources.add(resource);
                }
            }
        }
    }

    private String loadSpringConfiguration(final Filer filer, final String springConfigFilePath)
            throws IOException {
        try (InputStream inputStream =
                filer.getResource(StandardLocation.CLASS_OUTPUT, "", springConfigFilePath)
                        .openInputStream()) {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            int readByte;
            while ((readByte = inputStream.read()) != -1) {
                baos.write(readByte);
            }
            return baos.toString(StandardCharsets.UTF_8.name());
        }
    }

    public Element getType() {
        return type;
    }

    public Set<ResourceModel> getResources() {
        return Collections.unmodifiableSet(resources);
    }

    public String getClassName() {
        return className;
    }

    public String getName() {
        return name;
    }

    public void setName(final String name) {
        this.name = name;
    }

    public String getTags() {
        return tags;
    }

    public void setTags(final String tags) {
        this.tags = tags;
    }

    public String getBaseUrl() {
        return baseUrl;
    }

    public void setBaseUrl(final String baseUrl) {
        this.baseUrl = baseUrl;
    }

    public boolean isClient() {
        return client;
    }

    public void setClient(final boolean client) {
        this.client = client;
    }

    public boolean isXqAware() {
        return xqAware;
    }

    public void setXqAware(final boolean xqAware) {
        this.xqAware = xqAware;
    }

    public String getSpringConfig() {
        return springConfig;
    }

    public void setSpringConfig(final String springConfigFilePath) {
        this.springConfig = springConfigFilePath;
    }

    /**
     * <!--
     * ==================================================================================================
     * --> {@inheritDoc} <!--
     * -------------------------------------------------------------------------------------------------
     * -->
     */
    @Override
    public String toString() {
        return "ConnectModel [type=" + type + ", className=" + className + ", name=" + name
                + ", tags=" + tags + ", baseUrl=" + baseUrl + ", client=" + client + ", xqAware="
                + xqAware + ", springConfigFilePath=" + springConfig + ", resources=" + resources
                + "]";
    }

}
