// =====================================================================================================================
// Copyright (c) 2016. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.pojo.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * The {@linkplain SonicPOJO} annotation indicates that a class is intended to form a Sonic ESB Service. 
 * <p>
 * Annotation attributes
 * describe UI rendering, global settings of the ESB service type implemented by this POJO class.
 * <p>
 * Sonic POJO(Plain Old Java Object) Services allow Sonic services to be developed and unit tested without the need for
 * Sonic API expertise. Unit testing does not require a running domain, or, complex mocks for the Sonic APIs.
 * Similar to Spring, service initialization parameters translate to Java bean
 * properties "injected" by the environment at initialization time({@linkplain InitParameter} ). POJOs that
 * implement com.sonicsw.xq.XQServiceLifecycle can be informed on service lifecycle methods through this
 * interface. 
 * <p>
 * <strong>&#64;SonicPOJO</strong> ESB Service:
 *
 * <ol>
 * <li>must have at least one {@linkplain Operation} method</li>
 * <li>{@linkplain Operation}s must be public, must not be static, must have return type</li>
 * <li>fields that are annotated as {@linkplain InitParameter} must have public setter methods.</li>
 * </ol>
 *
 * <p>
 * Example usage:
 *
 * <pre>
 * import com.aurea.sonic.esb.pojo.annotation.InitParameter;
 * import com.aurea.sonic.esb.pojo.annotation.Operation;
 * import com.aurea.sonic.esb.pojo.annotation.Parameter;
 * import com.aurea.sonic.esb.pojo.annotation.SonicPOJO;
 *
 * import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
 *
 * <strong>&#64;SonicPOJO(description = "ConverterService POJO service")</strong>
 * public class ConverterService{
 *
 * 	private final Logger log = LogManager.getLogger(this.getClass());
 *
 * 	<strong>&#64;InitParameter(description = "Log operation calls", defaultValue = "false")</strong>
 * 	private boolean verbose;
 *
 * 	public void setVerbose(final boolean verbose) {
 * 		this.verbose = verbose;
 * 	}
 *
 * 	public boolean isVerbose() {
 * 		return verbose;
 * 	}
 *
 *	<strong>&#64;Operation(description = "Concatenate strings")</strong>
 *	public String concatenate(final String strList[], final String delimiter) {
 *		if (isVerbose()) {
 *			log.info("concatenate called with parameters strList={"+strList+"} and delimiter={"+delimiter+"}");
 *		}
 *		if (strList == null) {
 *			return "";
 *		}
 *		final StringBuffer sb = new StringBuffer();
 *
 *		for (int i = 0; i &lt; strList.length; i++) {
 *			sb.append(strList[i]);
 *			if (i &lt; strList.length - 1) {
 *				sb.append(delimiter);
 *			}
 *		}
 *		return sb.toString();
 *	}
 *
 *	<strong>&#64;Operation(description = "Split string")</strong>
 *	public String[] split(final String str, final String delimiter) {
 *		if (isVerbose()) {
 *			log.info("split called with parameters str={"+str+"} delimiter={"+delimiter+"}");
 *		}
 *		if (str == null) {
 *			return new String[0];
 *		}
 *		return str.split(delimiter);
 *	}
 *
 *	<strong>&#64;Operation(description = "Convert Fahrenheit To Celcius")</strong>
 *	public float convert(<strong>&#64;Parameter(contentType = "text/xml") final float fahrenheit</strong>) {
 *		if (isVerbose()) {
 *			log.info("convert called with parameter fahrenheitr={"+fahrenheit+"}");
 *		}
 *		return (fahrenheit - 32) * 5 / 9;
 *	}
 *
 * }
 * </pre>
 *
 * @since 10.0.7
 */
@Documented
@Inherited
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.CLASS)
public @interface SonicPOJO {

    /**
     * Service type name(default: the class simple name)
     * @return service type name
     *
     */
    String name() default "";

    /**
     * Service type display name(default: service name)
     * @return Display name
     *
     */
    String displayName() default "";

    /**
     * Name of default service instance created by upload(default: service name suffixed by "Instance")
     * @return instance name of sonic pojo service
     *
     */
    String instanceName() default "";

    /**
     * Service description(default: "")
     * @return descriptive text about the parameter usage and values
     */
    String description() default "";

}