// =====================================================================================================================
// Copyright (c) 2016. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.pojo.processor;

import java.util.List;
import java.util.Set;

import javax.annotation.processing.Messager;
import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeKind;
import javax.lang.model.util.ElementFilter;
import javax.tools.Diagnostic;

import com.aurea.sonic.esb.annotation.util.ProcessorContext;
import com.aurea.sonic.esb.pojo.processor.model.AttributeModel;
import com.aurea.sonic.esb.pojo.processor.model.OperationModel;
import com.aurea.sonic.esb.pojo.processor.model.ServiceModel;

/**
 * <!-- ========================================================================================================== -->
 * A validator that checks the {@linkplain ServiceModel} against predefined validation rules
 *
 * @since 10.0.7
 * <!-- --------------------------------------------------------------------------------------------------------- -->
 */
public class ESBServiceValidator {

	private final Messager messager;

	/**
	 * Constructs a new ESBServiceValidator from {@code context}
	 *
	 * @param context processor context
	 * @see ProcessorContext
	 */
	public ESBServiceValidator(final ProcessorContext context) {
		this.messager = context.getMessager();
	}

	/**
	 * <!-- ================================================================================================== -->
	 * This method validates the given {@code service} regarding to following rules.
	 *
	 * <p>
	 * Validation rules:
	 *
	 * <ol>
	 * <li>{@code service} must have at least one {@linkplain OperationModel}</li>
	 * <li>{@code service} attributes must have public setter methods.</li>
	 * <li>{@code service} operations must be public, must not be static, must have return type</li>
	 * </ol>
	 *
	 * @param service service model to be validated
	 * @return {@code true}  if service is valid, otherwise {@code false}
	 *
	 * <!-- ------------------------------------------------------------------------------------------------- -->
	 */
	public boolean validateService(final ServiceModel service) {
		boolean hasError = false;

		if (service.getOperations().isEmpty()) {
			addError(service.getType(), service.getName() + " service must have operations");
			hasError = true;
		}

		for (final AttributeModel attribute : service.getAttributes()) {
			if (!validateAttribute(attribute)) {
				hasError = true;
			}
		}

		for (final OperationModel operation : service.getOperations()) {
			if (!validateOperation(operation)) {
				hasError = true;
			}
		}

		return !hasError;
	}

	private boolean validateAttribute(final AttributeModel attribute) {
		if (!hasAttributeSetter(attribute)) {
			addError(attribute.getType(), attribute.getName() + " attribute must implement a public setter method");
			return false;
		}

		return true;
	}

	private boolean validateOperation(final OperationModel operation) {
		final ExecutableElement method = (ExecutableElement) operation.getType();

		final Set<Modifier> modifiers = method.getModifiers();

		if (!modifiers.contains(Modifier.PUBLIC)) {
			addError(method, operation.getName() + " operation must be public");
			return false;
		}

		if (modifiers.contains(Modifier.STATIC)) {
			addError(method, operation.getName() + " operation must not be static");
			return false;
		}

		if (method.getReturnType().getKind().equals(TypeKind.VOID)) {
			addError(method, operation.getName() + " operation must have return type");
			return false;
		}

		return true;
	}

	private boolean hasAttributeSetter(final AttributeModel attribute) {
		final String attributeSetter1 = "set" + upperFirstLetter(attribute.getName());
		final String attributeSetter2 = "set" + attribute.getName();
		final TypeElement attributeClass = (TypeElement) attribute.getType().getEnclosingElement();

		ExecutableElement attributeSetterElement = null;
		final List<ExecutableElement> classElements = ElementFilter.methodsIn(attributeClass.getEnclosedElements());
		for (final ExecutableElement e : classElements) {
			if (e.getSimpleName().toString().equals(attributeSetter1)
					|| e.getSimpleName().toString().equals(attributeSetter2)) {
				attributeSetterElement = e;
				break;
			}
		}

		return attributeSetterElement != null && attributeSetterElement.getModifiers().contains(Modifier.PUBLIC);
	}

	private String upperFirstLetter(final String str) {
		if (str.length() > 1) {
			return str.substring(0, 1).toUpperCase() + str.substring(1);
		} else if (str.length() == 1) {
			return str.toUpperCase();
		} else {
			return "";
		}
	}

	private void addError(final Element element, final String message, final Object... args) {
		final String formattedMessage = String.format(message, args);
		messager.printMessage(Diagnostic.Kind.ERROR, formattedMessage, element);
	}
}
