// =====================================================================================================================
// Copyright (c) 2016. Aurea Software, Inc. All Rights Reserved.
//
// You are hereby placed on notice that the software, its related technology and services may be covered by one or
// more United States ("US") and non-US patents. A listing that associates patented and patent-pending products
// included in the software, software updates, their related technology and services with one or more patent numbers
// is available for you and the general public's access at www.aurea.com/legal/ (the "Patent Notice") without charge.
// The association of products-to-patent numbers at the Patent Notice may not be an exclusive listing of associations,
// and other unlisted patents or pending patents may also be associated with the products. Likewise, the patents or
// pending patents may also be associated with unlisted products. You agree to regularly review the products-to-patent
// number(s) association at the Patent Notice to check for updates.
// =====================================================================================================================

package com.aurea.sonic.esb.pojo.processor.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.type.TypeKind;

import com.aurea.sonic.esb.annotation.util.ElementUtil;
import com.aurea.sonic.esb.annotation.util.ProcessorContext;
import com.aurea.sonic.esb.pojo.annotation.Operation;

/**
 * <!-- ========================================================================================================== -->
 * Data model of {@linkplain Operation} method
 * <!-- --------------------------------------------------------------------------------------------------------- -->
 */
public class OperationModel {

    private final Element type;

    private String name;

    private String description;

    private String contentType;

    private String outputType;

    private String outputName;

    private boolean isCollection;

    private final List<ParameterModel> parameters = new ArrayList<>();

    public OperationModel(final ProcessorContext context, final Element element) {
        this.type = element;

        final Operation operationAnnotation = element.getAnnotation(Operation.class);
        this.name = element.getSimpleName().toString();
        this.description = operationAnnotation.description();
        this.contentType = operationAnnotation.contentType();

        final ExecutableElement methodElement = (ExecutableElement) element;

        this.isCollection = methodElement.getReturnType().getKind() == TypeKind.ARRAY;

        final Class<?> javaType = ElementUtil.classFromType(context.getTypes(), methodElement.getReturnType());

        final String xmlType = ElementUtil.classToXmlType(javaType, isCollection);

        this.outputType = "".equals(operationAnnotation.outputType()) ? xmlType : operationAnnotation.outputType();

        this.outputName = operationAnnotation.outputName();

        for (final Element parameterElement : methodElement.getParameters()) {
            final ParameterModel parameter = new ParameterModel(context, parameterElement);
            addParameter(parameter);
        }
    }

    public Element getType() {
        return type;
    }

    public String getName() {
        return name;
    }

    public void setName(final String name) {
        this.name = name;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(final String description) {
        this.description = description;
    }

    public String getContentType() {
        return contentType;
    }

    public void setContentType(final String contentType) {
        this.contentType = contentType;
    }

    public String getOutputType() {
        return outputType;
    }

    public void setOutputType(final String outputType) {
        this.outputType = outputType;
    }

    public List<ParameterModel> getParameters() {
        return Collections.unmodifiableList(parameters);
    }

    public void addParameter(final ParameterModel parameter) {
        this.parameters.add(parameter);
    }

    public boolean isCollection() {
        return isCollection;
    }

    public void setCollection(final boolean isCollection) {
        this.isCollection = isCollection;
    }

    public String getOutputName() {
        return outputName;
    }

    public void setOutputName(final String outputName) {
        this.outputName = outputName;
    }

    @Override
    public String toString() {
        return "OperationType [type=" + type + ", name=" + name + ", description=" + description + ", contentType="
                + contentType + ", outputType=" + outputType + ", outputName=" + outputName + ", isCollection="
                + isCollection + ", parameters=" + parameters + "]";
    }

}
